/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

static const char __idstring[] = "@(#)$Id: mx_ether.c,v 1.46 2006/12/09 01:18:23 loic Exp $";

#include "mx_arch.h"
#include "mx_misc.h"
#include "mx_instance.h"
#include "mx_malloc.h"
#include "mx_ether_common.h"
#include "mx_stbar.h"
#include "mx_pio.h"

#include <netinet/in.h>
#include <netinet/in_var.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/ip_var.h>
#include <netinet/tcp.h>
#include <netinet/udp.h>

static inline int
mx_encap(struct mx_ether *eth, struct mbuf *m_head, mcp_kreq_ether_send_t *req)
{
	vm_paddr_t pa;
	struct mbuf *m;
	char *header;
	struct ip *ip;
	struct ether_header *eh;
	int i, idx;
	uint16_t cksum_offset;
	int cnt = 1;

	m = m_head;

	i = eth->tx.req;

	idx = i & (NUM_TX - 1);

	header = mtod(m_head, char *);

	/* Give the MCP the dest addr (already in network byte order) */
	req->head.dest_high16 = *(uint16_t *)&header[0];
	req->head.dest_low32 = *(uint32_t *)&header[2];	
	req->head.flags = MX_MCP_ETHER_FLAGS_HEAD | MX_MCP_ETHER_FLAGS_VALID;
	req->head.pseudo_hdr_offset = 0;
	req->head.cksum_offset = 0;

	/* save the mbuf in the last descriptor */
	eth->tx.info[idx].m = m_head;
	
	/* Setup checksum offloading, if needed */
	if (m->m_pkthdr.csum_flags & (CSUM_DELAY_DATA)) {
		/* ensure ip header is in first mbuf, copy
                   it to a scratch buffer if not */
                if (__predict_false(m->m_len < sizeof (*eh)
                                    + sizeof (*ip))) {
                        m_copydata(m, 0, sizeof (*eh) + sizeof (*ip),
                                   eth->arch.scratch);
                        eh = (struct ether_header *)eth->arch.scratch;
                } else {
                        eh = mtod(m, struct ether_header *);
                }
                ip = (struct ip *) (eh + 1);
		cksum_offset = sizeof(struct ether_header) + (ip->ip_hl << 2);
		req->head.pseudo_hdr_offset =
			htons(cksum_offset + m_head->m_pkthdr.csum_data);
		req->head.cksum_offset = htons(cksum_offset);
		req->head.flags = MX_MCP_ETHER_FLAGS_HEAD | MX_MCP_ETHER_FLAGS_CKSUM | MX_MCP_ETHER_FLAGS_VALID;
	}

	while (m) {
		req++;
		i++;
		cnt++;
		idx = i & (NUM_TX - 1);
		pa = vtophys(mtod(m, caddr_t));
		req->frag.addr_low = htonl(MX_LOWPART_TO_U32(pa));
		req->frag.addr_high = htonl(MX_HIGHPART_TO_U32(pa));
		req->frag.length = htons((uint16_t)m->m_len);
		m = m->m_next;

		/* This is used only for debugging */
		req->frag.flags = MX_MCP_ETHER_FLAGS_VALID;
	}

	/* terminate the descriptor */
	req->frag.flags |= MX_MCP_ETHER_FLAGS_LAST;
	return cnt;
}

static void
mx_ether_xmit(struct ifnet *ifp)
{
	mcp_kreq_ether_send_t req_list[MX_MCP_ETHER_MAX_SEND_FRAG + 1];
	struct mbuf *m_head, *m;
	struct mx_ether *eth = ifp->if_softc;
	int nsegs;

	mtx_lock(&eth->arch.tx_mtx);
	while (!IFQ_DRV_IS_EMPTY(&ifp->if_snd)) {
		/* dequeue the packet */
		IFQ_DRV_DEQUEUE(&ifp->if_snd, m_head);

		/* Count segments and make sure that the mbuf chain
		 * has <= MX_MCP_ETHER_MAX_SEND_FRAG segments.
		 */
		for (m = m_head, nsegs = 1; m != NULL; nsegs++) {
			if (__predict_false(m->m_len == 0)) {
				/* force a copy of the chain to strip
				   out zero-length mbufs */
				nsegs += MX_MCP_ETHER_MAX_SEND_FRAG;
			}
			m = m->m_next;
		}

		/* If too many segments, copy the packet */
		if (__predict_false(nsegs >= MX_MCP_ETHER_MAX_SEND_FRAG)) {
			m = m_defrag(m_head, M_DONTWAIT);
			if (m == NULL) {
				m_freem(m_head);
				ifp->if_oerrors++;
				continue;
			}
			m_head = m;
			for (m = m_head, nsegs = 1; m != NULL; nsegs++)
				m = m->m_next;
		}

		/* leave one slot to keep ring from wrapping */		
		if (__predict_false(nsegs > 
				    (NUM_TX - 1 - (eth->tx.req - eth->tx.done)))) {
			mx_ifp_set_active(ifp);
                        IFQ_DRV_PREPEND(&ifp->if_snd, m_head);
                        break;
                }

		/* feed it into the descriptor ring */
		mx_encap(eth, m_head, req_list);
		
		/* tell the MCP about it */
		mx_ether_submit_tx_req(eth, req_list, nsegs);

		BPF_MTAP(ifp, m_head);
	}
	mtx_unlock(&eth->arch.tx_mtx);	
}

void 
mx_ether_tx_done(mx_instance_state_t *is, uint32_t mcp_index)
{
	struct mx_ether *eth;
	struct mbuf *m;
	struct ifnet *ifp;
	int idx;

	eth = is->ether;
	ifp = eth->arch.ifp;


	while (eth->tx.done != mcp_index) {
		idx = eth->tx.done & (NUM_TX - 1);
		m = eth->tx.info[idx].m;

		/* Mark the DMA entry as free */
		eth->tx.info[idx].m = 0;
		eth->tx.done++;

		/* mbuf attached to only one slot in the packet */
		if (m) {
			m_freem(m);
			ifp->if_opackets++;
		}
	}

	/* If we have space, clear IFF_OACTIVE to tell the stack that
	   its OK to send packets */
	if (__predict_false((mx_ifp_is_active(ifp) &&
			     (eth->tx.req - eth->tx.done < (NUM_TX >> 2))))) {
		mx_ifp_clear_active(ifp);
		mx_ether_xmit(ifp);
	}
}

static inline int
mx_get_buf_big(struct mx_ether *eth, int idx, int how)
{
	vm_paddr_t pa;
	struct mbuf *m;
	mx_ether_rx_buf_t *rx = &eth->rx_big;

	m = m_getcl(how, MT_DATA, M_PKTHDR);
	if (!m) {
		eth->rx_big.alloc_fail++;
		return ENOBUFS;
	}
	rx->info[idx].m = m;
	pa = vtophys(mtod(m, caddr_t));
	rx->shadow[idx].addr_low = 
		htonl(MX_LOWPART_TO_U32(pa));
	rx->shadow[idx].addr_high = 
		htonl(MX_HIGHPART_TO_U32(pa));

	rx->cnt++;
	if ((idx & 7) == 7) {
		mx_pio_memcpy(&rx->ring[idx - 7], &rx->shadow[idx - 7],
			    8 * sizeof (*rx->ring), 0);
		MX_STBAR();
		MX_PIO_WRITE(rx->lanai_cnt, htonl(rx->cnt));
	}
	return 0;
}

void 
mx_ether_rx_done_big(mx_instance_state_t *is, int count, int len, 
		       int csum, int flags)
{
	struct mx_ether *eth;
	struct ifnet *ifp;
	struct mbuf *m = 0; 		/* -Wunitialized */
	struct mbuf *m_prev = 0;	/* -Wunitialized */
	struct mbuf *m_head = 0;
	mx_ether_rx_buf_t *rx;
	int idx;


	eth = is->ether;
	rx = &eth->rx_big;
	ifp = eth->arch.ifp;

	while (len > 0) {
		idx = rx->cnt & (NUM_RX - 1);
		/* save a pointer to the received mbuf */
		m = rx->info[idx].m;
		/* try to replace the received mbuf */
		if (__predict_false(mx_get_buf_big(eth, idx, M_NOWAIT))) {
			goto drop;
		}
		/* chain multiple segments together */
		if (!m_head) {
			m_prev = m_head = m;
			/* mcp implicitly skips 1st bytes so that
			 * packet is properly aligned */
			m->m_data += MX_MCP_ETHER_PAD;
			m->m_pkthdr.len = len;
			m->m_len = MCLBYTES - MX_MCP_ETHER_PAD;
		} else {
			m->m_len = MCLBYTES;
			m->m_flags &= ~M_PKTHDR;
			m_prev->m_next = m;
		}
		len -= m->m_len;
		m_prev = m;
	}

	/* trim trailing garbage from the last mbuf in the chain.  If
	 * there is any garbage, len will be negative */
	m->m_len += len;

	/* if the checksum is valid, mark it in the mbuf header */
	if (eth->csum_flag & flags) {
		m_head->m_pkthdr.csum_data = csum;
		m_head->m_pkthdr.csum_flags = CSUM_DATA_VALID;
	}
	
	/* pass the frame up the stack */
	m_head->m_pkthdr.rcvif = ifp;
	ifp->if_ipackets++;
	(*ifp->if_input)(ifp, m_head);
	return;
drop:

	/* drop the frame -- the old mbuf(s) are re-cycled */


	/* advance to the next frame, and give the DMA addresses
	   back to the MCP */
	do {
		idx = rx->cnt & (NUM_RX - 1);		
		rx->cnt++;
		if ((idx & 7) == 7) {
			mx_pio_memcpy(&rx->ring[idx - 7], &rx->shadow[idx - 7],
				    8 * sizeof (*rx->ring), 0);
			MX_STBAR();
			MX_PIO_WRITE(rx->lanai_cnt, htonl(rx->cnt));
		}
		
		len -= MCLBYTES;
	} while (len > 0);
	if (m_head)
		m_freem(m_head);
	ifp->if_ierrors++;

}

static inline int
mx_get_buf_small(struct mx_ether *eth, int idx, int how)
{
	vm_paddr_t pa;
	struct mbuf *m;
	mx_ether_rx_buf_t *rx = &eth->rx_small;
	int retval = 0;

	MGETHDR(m, how, MT_DATA);
	if (!m) {
		rx->alloc_fail++;
		retval = ENOBUFS;
		goto done;
	}
	rx->info[idx].m = m;
	pa = vtophys(mtod(m, caddr_t));
	rx->shadow[idx].addr_low = 
		htonl(MX_LOWPART_TO_U32(pa));
	rx->shadow[idx].addr_high = 
		htonl(MX_HIGHPART_TO_U32(pa));

done:
	if ((idx & 7) == 7) {
		mx_pio_memcpy(&rx->ring[idx - 7], &rx->shadow[idx - 7],
			    8 * sizeof (*rx->ring), 0);
		MX_STBAR();
		MX_PIO_WRITE(rx->lanai_cnt, htonl(rx->cnt));
	}
	return retval;
}

void 
mx_ether_rx_done_small(mx_instance_state_t *is, int count, int len, 
		       int csum, int flags)
{
	struct mx_ether *eth;
	struct ifnet *ifp;
	struct mbuf *m;
	int idx;

	eth = is->ether;
	ifp = eth->arch.ifp;
	idx = eth->rx_small.cnt & (NUM_RX - 1);
	eth->rx_small.cnt++;
	/* save a pointer to the received mbuf */
	m = eth->rx_small.info[idx].m;
	/* try to replace the received mbuf */
	if (mx_get_buf_small(eth, idx, M_DONTWAIT)) {
		/* drop the frame -- the old mbuf is re-cycled */
		ifp->if_ierrors++;
		return;
	}

	/* mcp implicitly skips 1st 2 bytes so that packet is properly
	 * aligned */
	m->m_data += MX_MCP_ETHER_PAD;

	/* if the checksum is valid, mark it in the mbuf header */
	if (eth->csum_flag & flags) {
		m->m_pkthdr.csum_data = csum;
		m->m_pkthdr.csum_flags = CSUM_DATA_VALID;
	}

	/* pass the frame up the stack */
	m->m_pkthdr.rcvif = ifp;
	m->m_len = m->m_pkthdr.len = len;
	ifp->if_ipackets++;
	(*ifp->if_input)(ifp, m);
}

static int
mx_ether_change_mtu(struct ifnet *ifp, int mtu)
{
	struct mx_ether *eth = ifp->if_softc;
	int error;

	if (mtu > MX_MAX_ETHER_MTU - ETHER_HDR_LEN) {
		error = EINVAL;
		goto done;
	}
	error = mx_mcpi.set_param(eth->is->id, eth->is->lanai.sram, 
				  "ethernet_mtu", mtu + ETHER_HDR_LEN);	
	MX_INFO(("%s: changing mtu from %d to %d\n",
		 ifp->if_xname, (int)ifp->if_mtu, mtu));
	if (!error)
		ifp->if_mtu = mtu;
done:
	return error;
}

static void
mx_ether_close(struct mx_ether *eth)
{
	struct mbuf *m;
	uint32_t dont_care;
	int i;

	/* if buffers not alloced, give up */
	if (!eth->rx_big.shadow)
		return;

	/* if the device not running give up */
	if (eth->running != MX_ETH_RUNNING &&
	    eth->running != MX_ETH_OPEN_FAILED)
		return;

	/* drop the ioctl lock so that we can sleep without
	   FreeBSD whining */
	eth->running = MX_ETH_STOPPING;
	mtx_unlock(&eth->arch.meta_mtx);

	mx_lanai_command(eth->is, MX_MCP_CMD_ETHERNET_DOWN,
			 0, 0, 0, &dont_care, &eth->cmd_sync);

	/* re-take the lock now that we've slept */
	mtx_lock(&eth->arch.meta_mtx);
	eth->running = 0;

	/* free recvs */
	for (i = 0; i < NUM_RX; i++) {
		eth->rx_small.shadow[i].addr_low = 0;
		m = eth->rx_small.info[i].m;
		eth->rx_small.info[i].m = 0;
		if (m)
			m_freem(m);

		eth->rx_big.shadow[i].addr_low = 0;
		m = eth->rx_big.info[i].m;
		eth->rx_big.info[i].m = 0;
		if (m)
			m_freem(m);
	}

	/* free transmits */
	for (i = 0; i < NUM_TX; i++) {
		m = eth->tx.info[i].m;
		eth->tx.info[i].m = 0;
		if (m)
			m_freem(m);
	}
	mx_ether_close_common(eth->is);
}

static int
mx_ether_open(struct mx_ether *eth)
{
	int error, i;
	uint32_t dont_care;
	struct ifnet *ifp;


	/* if the device not stopped, give up */
	if (eth->running != MX_ETH_STOPPED)
		return 0;

	ifp = eth->arch.ifp;
	
	eth->running = MX_ETH_STARTING;

	/* drop the lock.  Grr.. */
	mtx_unlock(&eth->arch.meta_mtx);

	error = mx_ether_open_common(eth->is, ifp->if_mtu + ETHER_HDR_LEN, 
				     MHLEN, MCLBYTES);
	if (error) {
		MX_WARN(("%s: mx_ether_open_common() failed, errno = %d\n",
			 ifp->if_xname, error));
		mtx_lock(&eth->arch.meta_mtx);
		goto abort_with_nothing;
	}

	/* allocate recvs */
	for (i = 0; i < NUM_RX; i++) {
		error = mx_get_buf_small(eth, i, M_WAITOK);
		if (error) {
			MX_WARN(("%s: Could not alloc small recv buffer %d, errno = %d\n",
				 ifp->if_xname, i, error));
			goto abort_with_open;
		}
		error = mx_get_buf_big(eth, i, M_WAITOK);
		if (error) {
			MX_WARN(("%s: Could not alloc big recv buffer %d, errno = %d\n",
				 ifp->if_xname, i, error));
			goto abort_with_open;
		}
	}

	eth->rx_small.cnt = NUM_RX;
	eth->rx_big.cnt = NUM_RX;
	MX_PIO_WRITE(eth->rx_small.lanai_cnt, htonl(eth->rx_small.cnt));
	MX_PIO_WRITE(eth->rx_big.lanai_cnt, htonl(eth->rx_big.cnt));

	/* somehow tell the mcp about this */
	error = mx_lanai_command(eth->is, MX_MCP_CMD_ETHERNET_UP,
				 0, 0, 0, &dont_care, &eth->cmd_sync);

	if (error) {
		MX_WARN(("%s: unable to start ethernet\n", ifp->if_xname));
		goto abort_with_open;
	}
	mx_ether_start_common(eth->is, ifp->if_mtu + ETHER_HDR_LEN, 
			     MHLEN, MCLBYTES);
	mtx_lock(&eth->arch.meta_mtx);
	eth->running = MX_ETH_RUNNING;
	return 0;
	
abort_with_open:
	eth->running = MX_ETH_OPEN_FAILED;
	mtx_lock(&eth->arch.meta_mtx);
	mx_ether_close(eth);
		
abort_with_nothing:
	eth->running = MX_ETH_STOPPED;
	return error;
}


static int
mx_media_change(struct ifnet *ifp)
{
	return EINVAL;
}

static void
mx_media_status(struct ifnet *ifp, struct ifmediareq *ifmr)
{
	struct mx_ether *eth = ifp->if_softc;

	if (eth == NULL)
		return;

	ifmr->ifm_status = IFM_AVALID;
	ifmr->ifm_status |= eth->is->link_state ? IFM_ACTIVE : 0;
	ifmr->ifm_active = IFM_AUTO | IFM_ETHER;
	ifmr->ifm_active |= eth->is->link_state ? IFM_FDX : 0;
}


static int
mx_ether_ioctl(struct ifnet *ifp, unsigned long cmd, caddr_t data)
{
	struct mx_ether *eth = ifp->if_softc;
	struct ifreq   *ifr = (struct ifreq *) data;
	int error, mask;

	mask = 0;
	error = 0;
	mtx_lock(&eth->arch.meta_mtx);
	switch (cmd) {
	case SIOCSIFADDR:
	case SIOCGIFADDR:
		error = ether_ioctl(ifp, cmd, data);
		break;

	case SIOCSIFMTU:
		error = mx_ether_change_mtu(ifp, ifr->ifr_mtu);
		break;

	case SIOCSIFFLAGS:
		if (ifp->if_flags & IFF_UP) {
			unsigned char *enaddr;

			if (!mx_ifp_is_running(ifp)) {
				error = mx_ether_open(eth);
				if (error)
					goto done;
				mx_ifp_set_running(ifp);
			}
			mx_ether_set_promisc_common(eth, 
						    ifp->if_flags & IFF_PROMISC);
			/* set the mac address if it was changed */
#if (__FreeBSD_version >= 600031)
			enaddr = IFP2ENADDR(ifp);
#else
			enaddr = eth->is_addr;
#endif

			if (bcmp (enaddr, eth->current_mac, 6)) {
				/* save the address */
				bcopy(enaddr, eth->current_mac, 6);
				/* set the addres on the LANai */
				mx_ether_set_mac_address_common(eth,
					eth->current_mac);
			}
		} else {
			if (mx_ifp_is_running(ifp)) {
				 mx_ether_close(eth);
				 mx_ifp_clear_running(ifp);
			}
		}
	case SIOCADDMULTI:
	case SIOCDELMULTI:
		error = 0;
		break;

	case SIOCSIFCAP:
		mask = ifr->ifr_reqcap ^ ifp->if_capenable;
		if (mask & IFCAP_TXCSUM) {
			if (IFCAP_TXCSUM & ifp->if_capenable) {
				ifp->if_capenable &= ~IFCAP_TXCSUM;
				ifp->if_hwassist &= ~(CSUM_TCP | CSUM_UDP);
                } else {
				ifp->if_capenable |= IFCAP_TXCSUM;
				ifp->if_hwassist |= (CSUM_TCP | CSUM_UDP);
			}
		} else if (mask & IFCAP_RXCSUM) {
			if (IFCAP_RXCSUM & ifp->if_capenable) {
				ifp->if_capenable &= ~IFCAP_RXCSUM;
				eth->csum_flag &= ~MX_MCP_ETHER_FLAGS_CKSUM;
                } else {
				ifp->if_capenable |= IFCAP_RXCSUM;
				eth->csum_flag |= MX_MCP_ETHER_FLAGS_CKSUM;
			}
		}
		break;

	case SIOCGIFMEDIA:
		error = ifmedia_ioctl(ifp, (struct ifreq *)data, 
				      &eth->arch.ifm, cmd);
		break;

	default:
		error = ENOTTY;
	}
done:
	mtx_unlock(&eth->arch.meta_mtx);
	/*printf("cmd = 0x%lx, returning %d\n", cmd, error);*/
	return error;
}


void
mx_ether_link_change_notify(mx_instance_state_t *is)
{
}

static void
mx_ether_init(void *eth)
{
	/* nothing */
}

int
mx_ether_attach(mx_instance_state_t *is)
{
	struct mx_ether *eth;
	struct ifnet *ifp;
	int i;

	eth = mx_kmalloc((sizeof *eth), MX_MZERO|MX_NOWAIT);
	if (!eth)
		return ENOMEM;
	eth->csum_flag = MX_MCP_ETHER_FLAGS_CKSUM;
	eth->is = is;
	is->ether = eth;

#if(__FreeBSD_version >= 600031)
	ifp = if_alloc(IFT_ETHER);
	if (ifp == NULL) {
		MX_WARN(("mx%d: Can't allocate ifp\n", is->id));
		return ENOMEM;
	}
#else
	ifp = &eth->is_if;
#endif
	eth->arch.ifp = ifp;
	mtx_init(&eth->arch.tx_mtx, "mx ether tx", NULL, MTX_DEF);
	mtx_init(&eth->arch.meta_mtx, "mx ether ioctl", NULL, MTX_DEF|MTX_RECURSE);
	ifp->if_softc = eth;
	for (i = 0; i < 6; i++) {
		eth->current_mac[i] = 
#if(__FreeBSD_version < 600031)
			eth->is_addr[i] = 
#endif
			is->mac_addr[i];
	}
	if_initname(ifp, "myri", is->id);
	ifp->if_hdrlen = sizeof (struct ether_header);
	ifp->if_timer = 0; /* XXXX */
	ifp->if_ioctl = mx_ether_ioctl;
	ifp->if_init = mx_ether_init;
	ifp->if_start = mx_ether_xmit;
	ifp->if_flags = IFF_BROADCAST | IFF_MULTICAST | IFF_SIMPLEX;
	ifp->if_baudrate = 1000000000;
	ifp->if_snd.ifq_maxlen = NUM_TX - 1; /* XXXX */
	ifp->if_snd.ifq_drv_maxlen = NUM_TX - 1;

	/* we support checksum offloading */
	ifp->if_capabilities = IFCAP_RXCSUM | IFCAP_TXCSUM;
	ifp->if_hwassist = CSUM_TCP | CSUM_UDP;
	ifp->if_capenable = ifp->if_capabilities;

	/* make the interface available to the system */
	ether_ifattach (ifp, eth->current_mac);

	/* ether_ifattach sets mtu to 1500 */
	ifp->if_mtu = MX_MAX_ETHER_MTU - ETHER_HDR_LEN;


	/* Initialise the ifmedia structure */
	ifmedia_init(&eth->arch.ifm, 0, mx_media_change, mx_media_status);
	ifmedia_add(&eth->arch.ifm, IFM_ETHER|IFM_AUTO, 0, NULL);
	return 0;
}

void
mx_ether_detach(mx_instance_state_t *is)
{
	struct mx_ether *eth;
	struct ifnet *ifp;

	eth = is->ether;
	if (!eth)
		return;
	ifp = eth->arch.ifp;
	ifmedia_removeall(&eth->arch.ifm);

	/* lock the device so that nobody can ifconfig it while
	   we're unloading */
	mtx_lock(&eth->arch.meta_mtx);
	if (eth->running) {
		mx_ether_close(eth);
	}
	ether_ifdetach(ifp);
	mtx_unlock(&eth->arch.meta_mtx);	

	/* the device is now gone, so we can free everything */
	mtx_destroy(&eth->arch.tx_mtx);
	mtx_destroy(&eth->arch.meta_mtx);
#if (__FreeBSD_version >= 600031)
	if_free(ifp);
#endif
	is->ether = 0;
	mx_kfree(eth);
}


int
mx_ether_parity_detach(mx_instance_state_t *is)
{
	struct mx_ether *eth;
	struct ifnet *ifp;

	eth = is->ether;
	if (!eth)
		return 0;
	ifp = eth->arch.ifp;

	if (ifp->if_flags & IFF_UP) {
		MX_WARN(("Detaching myri%d\n", is->id));
		mtx_lock(&eth->arch.meta_mtx);
		mx_ether_close(eth);
		mtx_unlock(&eth->arch.meta_mtx);
		return 1;
	}
	return 0;
}

void
mx_ether_parity_reattach(mx_instance_state_t *is)
{
	struct mx_ether *eth = is->ether;

	MX_WARN(("re-attaching myri%d\n", is->id));
	mtx_lock(&eth->arch.meta_mtx);
	mx_ether_open(eth);
	mtx_unlock(&eth->arch.meta_mtx);
}


/*
  This file uses MX driver indentation.

  Local Variables:
  c-file-style:"linux"
  tab-width:8
  End:
*/
